<?php

namespace Win7\Application\Manager;

use Win7\Application\Manager\DataTable\Column;
use Win7\Application\Manager\DataTable\Order;
use function ctype_digit;

/**
 * Representa os parâmetros de filtragem do sistema.
 *
 * @author Thiago Daher
 */
class FilterParams
{

    public const DEFAULT_LIMIT = 10;
    public const DEFAULT_START = 0;
    public const MAX_LIMIT = 100;

    /**
     * @var int
     */
    private int $limit;

    /**
     * @var int
     */
    private int $start;

    /**
     * @var int
     */
    private int $draw;

    /**
     * @var \Win7\Application\Manager\DataTable\Column[]
     */
    private array $columns;

    /**
     * @var \Win7\Application\Manager\DataTable\Order[]
     */
    private array $order;

    /**
     * @var array
     */
    private array $customSearch;

    /**
     * @var array
     */
    private array $options;

    /**
     * @param array $params
     */
    public function __construct(array $params)
    {
        $this->limit = $this->loadIntFromArray($params, 'length', self::DEFAULT_LIMIT);

        if ($this->limit > self::MAX_LIMIT || $this->limit <= 0) {
            $this->limit = self::DEFAULT_LIMIT;
        }

        $this->start = $this->loadIntFromArray($params, 'start', self::DEFAULT_START);
        $this->draw  = isset($params['draw']) && is_scalar($params['draw']) ?
            (int)$params['draw'] : 0;
        $this->loadColumns($params);
        $this->customSearch = [];
        $this->options      = $params;
    }

    /**
     * @param array $array
     * @param string $key
     * @param int $default
     * @return int
     */
    private function loadIntFromArray(array $array, string $key, int $default): int
    {
        $value = $array[$key] ?? $default;

        if (is_string($value) && (!ctype_digit($value) || strlen($value) > 5)) {
            $value = $default;
        }

        return (int)$value;
    }

    /**
     * @param array $params
     * @return void
     */
    private function loadColumns(array $params)
    {
        $this->columns = [];
        $this->order   = [];

        if (!isset($params['columns']) || !is_array($params['columns'])) {
            return;
        }

        $counter = 0;

        foreach ($params['columns'] as $column) {
            if (!is_array($column) ||
                !isset($column['name']) ||
                !is_string($column['name']) ||
                !preg_match('/^[\w\.]+$/', $column['name'])
            ) {
                continue;
            }

            $columnEntity = new Column();
            $columnEntity->setName($column['name']);
            $columnEntity->setPosition($counter);
            $this->columns[$counter] = $columnEntity;

            if (!isset($column['search']) ||
                !is_array($column['search']) ||
                !isset($column['search']['value']) ||
                $column['search']['value'] === ''
            ) {
                $counter++;
                continue;
            }

            $this->columns[$counter]->setSearch(trim($column['search']['value']));
            $counter++;
        }

        if (!isset($params['order']) || !is_array($params['order'])) {
            return;
        }

        foreach ($params['order'] as $order) {
            $column = (int)($order['column'] ?? 0);
            $dir    = isset($order['dir']) && $order['dir'] === 'asc' ? 'asc' : 'desc';

            if (!isset($this->columns[$column])) {
                continue;
            }

            $column = $this->columns[$column];
            $column->setOrder($dir);
            $order = new Order();
            $order->setColumn($column);
            $order->setDirection($dir);
            $this->order[] = $order;
        }
    }

    /**
     * @return int
     */
    public function getLimit(): int
    {
        return $this->limit;
    }

    /**
     * @return int
     */
    public function getStart(): int
    {
        return $this->start;
    }

    /**
     * @return int
     */
    public function getDraw(): int
    {
        return $this->draw;
    }

    /**
     * @return array
     */
    public function getColumns(): array
    {
        return $this->columns;
    }

    /**
     * @param array $columns
     */
    public function setColumns(array $columns): void
    {
        $this->columns = $columns;
    }

    /**
     * @return array
     */
    public function getOrder(): array
    {
        return $this->order;
    }

    /**
     * @param array $order
     */
    public function setOrder(array $order): void
    {
        $this->order = $order;
    }

    /**
     * @return array
     */
    public function getCustomSearch(): array
    {
        return $this->customSearch;
    }

    /**
     * @param array $customSearch
     */
    public function setCustomSearch(array $customSearch): void
    {
        $this->customSearch = $customSearch;
    }

    /**
     * @param string $column
     * @param $value
     * @return void
     */
    public function addCustomSearch(string $column, $value)
    {
        $this->customSearch[$column] = $value;
    }

    /**
     * @return array
     */
    public function getOptions(): array
    {
        return $this->options;
    }

    /**
     * @param array $options
     */
    public function setOptions(array $options): void
    {
        $this->options = $options;
    }

    /**
     * @param string $name
     * @param $default
     * @return mixed|null
     */
    public function getOption(string $name, $default = null)
    {
        return $this->options[$name] ?? $default;
    }

    /**
     * @param string $name
     * @param $value
     * @return mixed|null
     */
    public function setOption(string $name, $value)
    {
        return $this->options[$name] = $value;
    }

    /**
     * @return bool
     */
    public function isSearch(): bool
    {
        foreach ($this->getColumns() as $column) {
            if ($column->getSearch() !== '') {
                return true;
            }
        }

        return false;
    }

    /**
     * @return array
     */
    public function getCustomColumns(): array
    {
        $columns = [];

        foreach ($this->getColumns() as $column) {
            if (!$column->getDefinition()->isLoadFromDb()) {
                $columns[] = $column;
            }
        }

        return $columns;
    }

    /**
     * @param int $limit
     */
    public function setLimit(int $limit): void
    {
        $this->limit = $limit;
    }

    /**
     * @param int $start
     */
    public function setStart(int $start): void
    {
        $this->start = $start;
    }

    /**
     * @param int $draw
     */
    public function setDraw(int $draw): void
    {
        $this->draw = $draw;
    }

    /**
     * @param string $name
     * @return \Win7\Application\Manager\DataTable\Column|null
     */
    public function getColumn(string $name): ?Column
    {
        foreach ($this->getColumns() as $column) {
            if ($column->getName() === $name) {
                return $column;
            }
        }

        return null;
    }
}