<?php

namespace Win7\Application\Repository;

use Illuminate\Database\Connection;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Win7\Application\Entity\DatabaseEntity;
use Win7\Application\Manager\DataTable\Column;
use Win7\Application\Manager\DataTable\ColumnDefinition;
use Win7\Application\Manager\FilterParams;

/**
 * Representa um repositório: um serviço responsável pela comunicação
 * de leitura e escrita com o banco de dados
 *
 * @author Jerfeson Guerreiro <jerfeson_guerreiro@hotmail.com>
 * @author Thiago Daher
 */
abstract class Repository
{

    /**
     * @var \Win7\Application\Entity\DatabaseEntity
     */
    protected DatabaseEntity $entity;

    /**
     * @var \Win7\Application\Repository\RepositoryManager
     */
    private RepositoryManager $repositoryManager;

    /**
     * @var int
     */
    private int $tableCounter;

    /**
     * @var array
     */
    private array $joins;

    /**
     * Retorna o nome da classe da entidade.
     *
     * @return string
     */
    abstract public function getEntityClass(): string;

    /**
     * @param \Win7\Application\Entity\DatabaseEntity $entity
     * @return void
     */
    public function setEntity(DatabaseEntity $entity): void
    {
        $this->entity = $entity;
    }

    /**
     * Salva a entidade no banco de dados.
     *
     * @param DatabaseEntity $entity
     */
    public function save(DatabaseEntity $entity)
    {
        $entity->save();
    }

    /**
     * Insere uma nova entidade no banco de dados através da array.
     *
     * @param array $item
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function insert(array $item): Builder
    {
        return $this->newQuery()->create($item);
    }

    /**
     * Cria uma nova query.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    protected function newQuery(): Builder
    {
        return $this->entity->newQuery();
    }

    /**
     * Encontra a entidade com o ID especificado.
     *
     * @param $id
     * @return DatabaseEntity|null|mixed
     */
    public function findById($id): ?DatabaseEntity
    {
        return $this->newQuery()->find($id);
    }

    /**
     * @param FilterParams $params
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public function getTableList(FilterParams $params)
    {
        $query = $this->getTableQuery($params);

        if ($params->getLimit() !== -1) {
            $query->limit($params->getLimit());
            $query->offset($params->getStart());
        }

        return $query->get();
    }

    /**
     * @param FilterParams $params
     * @param bool $select
     * @return \Illuminate\Database\Eloquent\Builder
     */
    protected function getTableQuery(FilterParams $params, bool $select = true): Builder
    {
        $query              = $this->newQuery();
        $this->tableCounter = 0;
        $this->joins        = [];

        foreach ($params->getColumns() as $column) {
            $this->processColumn($query, $column, $select);
        }

        foreach ($params->getOrder() as $order) {
            $column = $order->getColumn();

            if (!$column->getDefinition()->isLoadFromDb()) {
                continue;
            }

            $columnName = $this->getQueryColumnName($query, $column);
            $query->orderBy($columnName, $order->getDirection());
        }

        foreach ($params->getCustomSearch() as $column => $value) {
            $isSub = strpos($column, '.') !== false;

            if ($isSub) {
                $this->doTableJoin($query, $column, ColumnDefinition::INNER_JOIN);
            }

            if (is_callable($value)) {
                $value($query);
            } else {
                $query->where($column, '=', $value);
            }
        }

        if (isset($query->from)) {
            $query->orderBy($query->from . '.id', 'desc');
        }

        return $query;
    }

    /**
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param \Win7\Application\Manager\DataTable\Column $column
     * @return string
     */
    private function getQueryColumnName(Builder $query, Column $column): string
    {
        $tableData = explode('.', $column->getName());

        if (strpos($column->getName(), '.') !== false) {
            return $tableData[count($tableData) - 2] . '.' . $tableData[count($tableData) - 1];
        }

        return "$query->from.{$column->getName()}";
    }

    /**
     * @param Builder $query
     * @param Column $column
     * @param bool $select
     * @return void
     */
    private function processColumn(Builder $query, Column $column, bool $select = true)
    {
        if (!$column->getDefinition()->isLoadFromDb()) {
            return;
        }

        $columnName = $this->getQueryColumnName($query, $column);

        if (strpos($column->getName(), '.') !== false) {
            $this->doTableJoin($query, $column->getName(), $column->getDefinition()->getJoin());
        }

        if ($select) {
            $query->addSelect($columnName . ' as column_' . $column->getPosition());
        }

        $this->tableCounter++;

        if ($column->getSearch() === '') {
            return;
        }

        if ($column->getSearchType() === ColumnDefinition::SEARCH_EQUAL) {
            $query->where($columnName, '=', $column->getSearch());
        } elseif ($column->getSearchType() === ColumnDefinition::SEARCH_LIKE_END_ONLY) {
            $query->where($columnName, 'like', $column->getSearch() . '%');
        } else {
            $query->where($columnName, 'like', '%' . $column->getSearch() . '%');
        }
    }

    /**
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $columnName
     * @param int $joinType
     * @return void
     */
    protected function doTableJoin(Builder $query, string $columnName, int $joinType)
    {
        $tableData = explode('.', $columnName);
        $entity    = $this->entity;
        $method    = $joinType === ColumnDefinition::INNER_JOIN ? 'join' : 'leftJoin';

        for ($counter = 0; $counter < count($tableData) - 1; $counter++) {
            $table = $tableData[$counter];

            if (!method_exists($entity, $table)) {
                break;
            }

            $relation = $entity->$table();
            $tableKey = implode('.', array_slice($tableData, 0, $counter + 1));

            if (in_array($tableKey, $this->joins)) {
                $entity = $relation->getRelated();
                continue;
            }

            if ($relation instanceof BelongsTo) {
                $realTable = $relation->getRelated()->getTable();
                $query->$method(
                    $realTable . ' as ' . $table,
                    $entity->getTable() . '.' . $relation->getForeignKeyName(),
                    '=',
                    $table . '.id'
                );
            } elseif ($relation instanceof HasMany) {
                $realTable = $relation->getRelated()->getTable();
                $query->$method(
                    $realTable . ' as ' . $table,
                    $table . '.' . $relation->getForeignKeyName(),
                    '=',
                    $entity->getTable() . '.id'
                );
                $query->groupBy($query->from . '.id');
            } else {
                break;
            }

            $entity        = $relation->getRelated();
            $this->joins[] = $tableKey;
        }
    }

    /**
     * @param FilterParams $params
     * @return int
     */
    public function getTableCount(FilterParams $params): int
    {
        $query = $this->getTableQuery($params, false);

        return $query->count();
    }

    /**
     * Encontra as entidades através dos critérios especificados.
     *
     * @param array $params
     * @param array $with
     * @return \Illuminate\Database\Eloquent\Collection|object[]
     */
    public function findBy(array $params, array $with = [])
    {
        return $this->queryWhere($params, $with)->get();
    }

    /**
     * Criar uma nova query com os parâmetros de filtragem especificados.
     *
     * @param array $params
     * @param array $with
     * @return \Illuminate\Database\Eloquent\Builder
     */
    protected function queryWhere(array $params, array $with = []): Builder
    {
        $query = $this->newQuery();

        foreach ($params as $key => $value) {
            $query->where($key, '=', $value);
        }

        if (!empty($with)) {
            $query->with($with);
        }

        return $query;
    }

    /**
     * Encontra uma entidade com os dados especificados.
     *
     * @param array $params
     * @param array $with
     * @return DatabaseEntity|null
     */
    public function findOneBy(array $params, array $with = []): ?object
    {
        return $this->queryWhere($params, $with)->limit(1)->get()->first();
    }

    /**
     * @return \Win7\Application\Repository\RepositoryManager
     */
    public function getRepositoryManager(): RepositoryManager
    {
        return $this->repositoryManager;
    }

    /**
     * @param \Win7\Application\Repository\RepositoryManager $repositoryManager
     */
    public function setRepositoryManager(RepositoryManager $repositoryManager): void
    {
        $this->repositoryManager = $repositoryManager;
    }

    /**
     * @return \Illuminate\Database\Connection
     */
    protected function getConnection(): Connection
    {
        return $this->entity->getConnection();
    }

    /**
     * @param array $params
     * @return bool
     */
    public function entityExists(array $params): bool
    {
        $query = $this->newQuery();

        foreach ($params as $key => $value) {
            $query->where($key, '=', $value);
        }

        return $query->exists();
    }
}
