<?php

namespace Win7\Phone\DataTable;

use Win7\Application\Common\StringUtils;
use Win7\Application\Manager\DataTable\ColumnDefinition;
use Win7\Application\Manager\FilterParams;
use Win7\Application\Manager\TableManager;
use Win7\Phone\Repository\CallRepository;

/**
 * @namespace Win7\Phone\DataTable
 * @author Thiago Daher
 *
 * @method CallRepository getRepository()
 */
class CallTable extends TableManager
{

    /**
     * @return string
     */
    protected function getRepositoryClass(): string
    {
        return CallRepository::class;
    }

    /**
     * @return array|\Win7\Application\Manager\DataTable\ColumnDefinition[]
     */
    protected function getColumnDefinitions(): array
    {
        return [
            'id' => new ColumnDefinition([
                'position'    => 0,
                'orderable'   => false,
                'searchable'  => false,
                'search_type' => ColumnDefinition::SEARCH_EQUAL,
            ]),

            'start_time' => new ColumnDefinition([
                'position'    => 1,
                'orderable'   => true,
                'searchable'  => true,
                'search_type' => ColumnDefinition::SEARCH_LIKE,
            ]),

            'source' => new ColumnDefinition([
                'position'    => 2,
                'orderable'   => true,
                'searchable'  => true,
                'search_type' => ColumnDefinition::SEARCH_LIKE,
            ]),

            'destination' => new ColumnDefinition([
                'position'    => 3,
                'orderable'   => true,
                'searchable'  => true,
                'search_type' => ColumnDefinition::SEARCH_LIKE,
            ]),

            'category' => new ColumnDefinition([
                'position'    => 4,
                'orderable'   => true,
                'searchable'  => true,
                'search_type' => ColumnDefinition::SEARCH_LIKE,
            ]),

            'real_cost' => new ColumnDefinition([
                'position'    => 5,
                'orderable'   => true,
                'searchable'  => true,
                'search_type' => ColumnDefinition::SEARCH_LIKE,
                'callback'    => [StringUtils::class, 'formatMoney'],
            ]),

            'real_talking_duration' => new ColumnDefinition([
                'position'    => 6,
                'orderable'   => true,
                'searchable'  => true,
                'search_type' => ColumnDefinition::SEARCH_LIKE_END_ONLY,
            ]),

            'real_ringing_duration' => new ColumnDefinition([
                'position'    => 7,
                'orderable'   => true,
                'searchable'  => true,
                'search_type' => ColumnDefinition::SEARCH_LIKE_END_ONLY,
            ]),

            'real_total_duration' => new ColumnDefinition([
                'position'    => 8,
                'orderable'   => true,
                'searchable'  => true,
                'search_type' => ColumnDefinition::SEARCH_LIKE_END_ONLY,
            ]),

            'adicionado' => new ColumnDefinition([
                'position'     => 9,
                'orderable'    => false,
                'searchable'   => false,
                'load_from_db' => false,
                'callback'     => [$this, 'setAdicionado'],
            ]),

            'answered' => new ColumnDefinition([
                'position'    => 10,
                'orderable'   => true,
                'searchable'  => true,
                'search_type' => ColumnDefinition::SEARCH_LIKE,
            ]),

            'valido' => new ColumnDefinition([
                'position'     => 11,
                'orderable'    => false,
                'searchable'   => false,
                'search_type'  => ColumnDefinition::SEARCH_LIKE,
                'load_from_db' => false,
                'callback'     => [$this, 'setValido'],
            ]),

            'call_quantity' => new ColumnDefinition([
                'position'     => 12,
                'orderable'    => false,
                'searchable'   => false,
                'search_type'  => ColumnDefinition::SEARCH_LIKE,
                'load_from_db' => false,
                'callback'     => [$this, 'setQuantity'],
            ]),

            'placeholder' => new ColumnDefinition([
                'position'     => 13,
                'orderable'    => false,
                'searchable'   => false,
                'search_type'  => ColumnDefinition::SEARCH_LIKE,
                'load_from_db' => false,
            ]),
        ];
    }

    /**
     * @param array $context
     * @return bool
     * @throws \ReflectionException
     */
    public function setAdicionado(array $context): bool
    {
        return $this->getRepository()->hasContatos(
            $context['source'],
            $this->getParams()->getOption('cliente')
        );
    }

    /**
     * @param array $context
     * @return bool
     */
    public function setValido(array $context): bool
    {
        return $context['source'] !== $context['source_caller_id'];
    }

    /**
     * @param array $context
     * @return int
     */
    public function setQuantity(array $context): int
    {
        return $context['call_quantity'] ?? 1;
    }

    /**
     * @param \Win7\Application\Manager\FilterParams $params
     * @return array
     * @throws \ReflectionException
     * @throws \Win7\Application\Exception\UserException
     */
    public function getTableList(FilterParams $params): array
    {
        $result          = parent::getTableList($params);
        $showTotal       = $params->getOption('carregarTotais') == 1;
        $result['valor'] = 'R$ 0,00';
        $result['total'] = $result['recordsTotal'];

        if ($showTotal) {
            $result['valor'] = StringUtils::formatMoney($this->getRepository()->getCallCost($params));
        }

        if ($showTotal && $params->getOption('agrupar') && $params->getOption('totalReal')) {
            $params->setOption('agrupar', false);
            $result['total'] = $this->getRepository()->getTableCount($params);
        }

        $result['permitirLeads'] = $this->getParams()->getOption('permitirLeads');
        $result['urlContato']    = $this->getParams()->getOption('urlContato');

        return $result;
    }
}