<?php

namespace Win7\Phone\Repository;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Win7\Application\Manager\FilterParams;
use Win7\Application\Repository\Repository;
use Win7\Phone\Entity\Call;

/**
 * @namespace Win7\Phone\Repository
 * @author Thiago Daher
 *
 * @method Call|null findById($id)
 */
class CallRepository extends Repository
{

    /**
     * @var \Win7\Application\Manager\FilterParams
     */
    private FilterParams $params;

    /**
     * @return string
     */
    public function getEntityClass(): string
    {
        return Call::class;
    }

    /**
     * @return \Win7\Application\Manager\FilterParams
     */
    public function getParams(): FilterParams
    {
        return $this->params;
    }

    /**
     * @param \Win7\Application\Manager\FilterParams $params
     */
    public function setParams(FilterParams $params): void
    {
        $this->params = $params;
    }

    /**
     * @param \Win7\Application\Manager\FilterParams $params
     * @param bool $select
     * @return \Illuminate\Database\Eloquent\Builder
     */
    protected function getTableQuery(FilterParams $params, bool $select = true): Builder
    {
        $this->setParams($params);
        $query = parent::getTableQuery($params, $select);

        if ($select) {
            $query->addSelect('call.source_caller_id as source_caller_id');
        }

        $query->orderBy('call.start_time', 'desc');
        $query->where('call.start_time', '>=', $params->getOption('dataInicial'));
        $query->where('call.start_time', '<=', $params->getOption('dataFinal'));
        $query->where('call.cliente_id', '=', $params->getOption('cliente'));

        if ($params->getOption('agrupar')) {
            $query->groupBy('call.source');
            $query->selectRaw('count(distinct call.id) as call_quantity');
        }

        return $query;
    }

    /**
     * @param string $numero
     * @param string $idCliente
     * @return bool
     */
    public function hasContatos(string $numero, string $idCliente): bool
    {
        $query = $this->getConnection()->query();
        $query->select('NU_TELEFONE1');
        $query->where('ID_CLIENTE_FK', '=', $idCliente);
        $query->where('NU_TELEFONE1', '=', $numero);
        $query->where('TXT_LEADTIPO', '=', 'Win7 Fone');
        $query->from('TBL_LEADS');
        $query->limit(1);

        return $query->exists();
    }

    /**
     * @param \Win7\Application\Manager\FilterParams $params
     * @return float
     */
    public function getCallCost(FilterParams $params): float
    {
        $query = $this->getCallsQuery($params);
        $query->selectRaw('SUM(real_cost) as cost');
        $queryResult = $query->get()->first();

        return $queryResult->cost ?? 0;
    }

    /**
     * @param \Win7\Application\Manager\FilterParams $params
     * @return \Illuminate\Database\Query\Builder
     */
    private function getCallsQuery(FilterParams $params): \Illuminate\Database\Query\Builder
    {
        $this->setParams($params);
        $query = $this->getConnection()->query();
        $query->from('call');
        $query->where('call.start_time', '>=', $params->getOption('dataInicial'));
        $query->where('call.start_time', '<=', $params->getOption('dataFinal'));
        $query->where('call.cliente_id', '=', $params->getOption('cliente'));

        return $query;
    }

    /**
     * @param \Win7\Application\Manager\FilterParams $params
     * @return \Illuminate\Support\Collection
     */
    public function findSubligacoes(FilterParams $params): Collection
    {
        $query = $this->getConnection()->query();
        $query->select('call.start_time');
        $query->addSelect('call.real_cost');
        $query->addSelect('call.id');
        $query->from('call');
        $query->where('call.start_time', '>=', $params->getOption('dataInicial'));
        $query->where('call.start_time', '<=', $params->getOption('dataFinal'));
        $query->where('call.cliente_id', '=', $params->getOption('cliente'));
        $query->where('call.id', '!=', $params->getOption('id'));
        $query->where('call.source', '=', $params->getOption('source'));

        return $query->get();
    }

    /**
     * @param \Win7\Application\Manager\FilterParams $params
     * @return int
     */
    public function countCalls(FilterParams $params): int
    {
        $query = $this->getCallsQuery($params);

        if ($params->getOption('atendidas') !== null) {
            $query->where('call.answered', '=', $params->getOption('atendidas') ? '1' : '0');
        }

        if ($params->getOption('agrupar') && !$params->getOption('totalReal')) {
            $query->select('id');
            $query->groupBy('source');

            return $this->getGroupedQuantity($query->toSql(), $query->getBindings());
        }

        $query->selectRaw('COUNT(*) as quantity');
        $queryResult = $query->get()->first();

        return $queryResult->quantity ?? 0;
    }

    /**
     * @param string $startDate
     * @param string $endDate
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection|\Win7\Phone\Entity\Call[]
     */
    public function findCallsByPeriod(string $startDate, string $endDate)
    {
        $query = $this->newQuery();
        $query->where('call.start_time', '>=', $startDate);
        $query->where('call.start_time', '<=', $endDate);

        /** @var Call[]|\Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection $result */
        $result = $query->get();

        return $result;
    }

    /**
     * @param \Win7\Application\Manager\FilterParams $params
     * @return int
     */
    public function getTableCount(FilterParams $params): int
    {
        if (!$params->getOption('agrupar')) {
            return parent::getTableCount($params);
        }

        $query = $this->getTableQuery($params, false);
        $query->select('id');
        $query->from('call');
        $query->where('call.start_time', '>=', $params->getOption('dataInicial'));
        $query->where('call.start_time', '<=', $params->getOption('dataFinal'));
        $query->where('call.cliente_id', '=', $params->getOption('cliente'));
        $query->groupBy('source');

        return $this->getGroupedQuantity($query->toSql(), $query->getBindings());
    }

    /**
     * @param string $query
     * @param array $bindings
     * @return int
     */
    private function getGroupedQuantity(string $query, array $bindings): int
    {
        $sql = sprintf(
            "SELECT %s FROM (%s) as dados",
            'COUNT(id) as quantity',
            $query
        );

        $result = $this->getConnection()->select($sql, $bindings);
        $result = !empty($result) ? $result[0] : [];

        return $result->quantity ?? 0;
    }
}