<?php

namespace Win7\Ads\Report;

use Win7\Ads\Exception\RuntimeException;

/**
 * Classe utilitária para montar as queries de consulta do google ads.
 *
 * @author Thiago Daher
 */
class QueryBuilder
{

    /**
     * @var array
     */
    private $select;

    /**
     * @var array
     */
    private $where;

    /**
     * @var string
     */
    private $from;

    /**
     * @var array
     */
    private $orderBy;

    /**
     * @var \Win7\Ads\Report\QueryExpressionBuilder
     */
    private $expr;

    /**
     * Construtor.
     */
    public function __construct()
    {
        $this->clear();
        $this->expr = new QueryExpressionBuilder();
    }

    /**
     * @return void
     */
    public function clear()
    {
        $this->select  = [];
        $this->from    = '';
        $this->where   = [];
        $this->orderBy = [];
    }

    /**
     * Retorna o montador de expressões.
     *
     * @return \Win7\Ads\Report\QueryExpressionBuilder
     */
    public function expr(): QueryExpressionBuilder
    {
        return $this->expr;
    }

    /**
     * Limpa todas as seleções e seleciona o valor especificado.
     *
     * @param string $select
     * @return void
     */
    public function select(string $select)
    {
        $this->select = [$select];
    }

    /**
     * Limpa todas as condições e adiciona uma condição na query.
     *
     * @param string $where
     * @return void
     */
    public function where(string $where)
    {
        $this->where = [$where];
    }

    /**
     * Adiciona uma condição na query.
     *
     * @param string $where
     * @return void
     */
    public function andWhere(string $where)
    {
        $this->where[] = $where;
    }

    /**
     * Limpa todas as ordenações e adiciona a ordenação especificada.
     *
     * @param string $orderBy
     * @return void
     */
    public function orderBy(string $orderBy)
    {
        $this->orderBy = [$orderBy];
    }

    /**
     * Adiciona a ordenação especificada.
     *
     * @param string $orderBy
     * @return void
     */
    public function andOrderBy(string $orderBy)
    {
        $this->orderBy[] = $orderBy;
    }

    /**
     * Adiciona a seleção especificada.
     *
     * @param string $select
     * @return void
     */
    public function andSelect(string $select)
    {
        $this->select[] = $select;
    }

    /**
     * Indica de qual recurso do ads a query selecionará os dados.
     *
     * @param string $from
     * @return void
     */
    public function from(string $from)
    {
        $this->from = $from;
    }

    /**
     * Retorna a representação textual da query.
     *
     * @return string
     */
    public function asString(): string
    {
        if (empty($this->select) || empty($this->from)) {
            throw new RuntimeException('Query inválida');
        }

        $result = 'SELECT ' . implode(', ', $this->select);
        $result .= ' FROM ' . $this->from;

        if (!empty($this->where)) {
            $result .= ' WHERE ' . implode(' AND ', $this->where);
        }

        if (!empty($this->orderBy)) {
            $result .= ' ORDER BY ' . implode(', ', $this->orderBy);
        }

        return $result;
    }

    /**
     * Verifica se a query é válida.
     *
     * @return bool
     */
    public function isValid(): bool
    {
        return !empty($this->select) && !empty($this->from);
    }
}