<?php

namespace Win7\Ads\Entity;

use DateTime;

/**
 * Objeto que guarda as informações de filtragem de campanha do cliente.
 *
 * @author Thiago Daher
 */
class CustomerFilter
{

    /**
     * Indica que todas as campanhas devem ser exibidas.
     */
    public const MODE_SHOW_ALL = 0;

    /**
     * Indica que apenas algumas campanhas devem ser exibidas.
     */
    public const MODE_SHOW_ONLY_CAMPAIGNS = 1;

    /**
     * Indica que todas as campanhas devem ser exibidas, exceto algumas
     */
    public const MODE_SHOW_ALL_EXCEPT_CAMPAIGNS = 2;

    /**
     * Lista de modos de filtragem de campanha disponíveis
     */
    public const AVAILABLE_MODES = [
        self::MODE_SHOW_ALL,
        self::MODE_SHOW_ONLY_CAMPAIGNS,
        self::MODE_SHOW_ALL_EXCEPT_CAMPAIGNS,
    ];

    /**
     * ID da conta
     *
     * @var string
     */
    private $customerId;

    /**
     * Modo de filtragem de campanhas
     *
     * @var string
     */
    private $campaignMode = self::MODE_SHOW_ALL;

    /**
     * Lista de IDs das campanhas que serão filtradas
     *
     * @var array
     */
    private $campaignIds = [];

    /**
     * Tipo de filtro de data
     *
     * @var string
     */
    private $dateFilter = '';

    /**
     * Data inicial de filtragem
     *
     * @var string
     */
    private $startDate = '';

    /**
     * Data final de filtragem
     *
     * @var string
     */
    private $endDate = '';

    /**
     * Indica se devem ser exibidas apenas campanhas de pesquisa
     *
     * @var bool
     */
    private $showOnlySearch = true;

    /**
     * Indica se é possível pausar campanhas
     *
     * @var bool
     */
    private $campaignPauseAllowed = true;

    /**
     * Data mínima de filtragem, para o caso de ser o filtro "Desde o início"
     *
     * @var string
     */
    private $minimumDate = '2019-01-01';

    /**
     * @return string
     */
    public function getCampaignMode(): string
    {
        return $this->campaignMode;
    }

    /**
     * @param mixed $campaignMode
     */
    public function setCampaignMode($campaignMode): void
    {
        if (is_string($campaignMode) && is_numeric($campaignMode)) {
            $campaignMode = (int)$campaignMode;
        }

        $this->campaignMode = is_int($campaignMode) && in_array($campaignMode, self::AVAILABLE_MODES) ?
            $campaignMode : self::MODE_SHOW_ALL;
    }

    /**
     * @return array
     */
    public function getCampaignIds(): array
    {
        return $this->campaignIds;
    }

    /**
     * @param mixed $campaignIds
     */
    public function setCampaignIds($campaignIds): void
    {
        $this->campaignIds = is_array($campaignIds) ? $campaignIds : [];
    }

    /**
     * @return string
     */
    public function getDateFilter(): string
    {
        return $this->dateFilter;
    }

    /**
     * @param string $dateFilter
     * @return void
     * @throws \Exception
     */
    public function setDateFilter(string $dateFilter): void
    {
        switch ($dateFilter) {
            case "LAST_7_DAYS_TODAY":
                $this->dateFilter = 'CUSTOM_DATE';
                $this->startDate  = date("Y-m-d", strtotime("-7 days"));
                $this->endDate    = date("Y-m-d");
                break;
            case "LAST_14_DAYS_TODAY":
                $this->dateFilter = 'CUSTOM_DATE';
                $this->startDate  = date("Y-m-d", strtotime("-14 days"));
                $this->endDate    = date("Y-m-d");
                break;
            case "LAST_30_DAYS_TODAY":
                $this->dateFilter = 'CUSTOM_DATE';
                $this->startDate  = date("Y-m-d", strtotime("-30 days"));
                $this->endDate    = date("Y-m-d");
                break;
            case "ALL_TIME":
                $this->dateFilter = 'CUSTOM_DATE';
                $this->startDate  = $this->minimumDate;
                $this->endDate    = date("Y-m-d");
                break;
            default:
                $this->dateFilter = $dateFilter;
        }
    }

    /**
     * @return string
     */
    public function getStartDate(): string
    {
        return $this->startDate;
    }

    /**
     * @param string $startDate
     * @return void
     * @throws \Exception
     */
    public function setStartDate(string $startDate): void
    {
        if (empty($startDate)) {
            return;
        }

        $this->startDate = $this->convertUserDateToAdsDate($startDate);
    }

    /**
     * @param string $date
     * @return string
     * @throws \Exception
     */
    protected function convertUserDateToAdsDate(string $date): string
    {
        $date = str_replace(['%2F'], '/', $date);
        $date = str_replace(['/'], '-', $date);
        $date = new DateTime($date);

        if (empty($date)) {
            return '';
        }

        return $date->format('Y-m-d');
    }

    /**
     * @return string
     */
    public function getEndDate(): string
    {
        return $this->endDate;
    }

    /**
     * @param string $endDate
     * @return void
     * @throws \Exception
     */
    public function setEndDate(string $endDate): void
    {
        if (empty($endDate)) {
            return;
        }

        $this->endDate = $this->convertUserDateToAdsDate($endDate);
    }

    /**
     * @return bool
     */
    public function isShowOnlySearch(): bool
    {
        return $this->showOnlySearch;
    }

    /**
     * @param bool $showOnlySearch
     */
    public function setShowOnlySearch(bool $showOnlySearch): void
    {
        $this->showOnlySearch = $showOnlySearch;
    }

    /**
     * @return string
     */
    public function getCustomerId(): string
    {
        return $this->customerId;
    }

    /**
     * @param string $customerId
     */
    public function setCustomerId(string $customerId): void
    {
        $this->customerId = preg_replace('/[^0-9]/', '', $customerId);
    }

    /**
     * @return bool
     */
    public function isCampaignPauseAllowed(): bool
    {
        return $this->campaignPauseAllowed;
    }

    /**
     * @param mixed $campaignPauseAllowed
     */
    public function setCampaignPauseAllowed($campaignPauseAllowed): void
    {
        $this->campaignPauseAllowed = (
            $campaignPauseAllowed ||
            ((int)$campaignPauseAllowed) === 1
        );
    }

    /**
     * @return string
     */
    public function getMinimumDate(): string
    {
        return $this->minimumDate;
    }

    /**
     * @param string $minimumDate
     * @return void
     * @throws \Exception
     */
    public function setMinimumDate(string $minimumDate): void
    {
        $minimumDate = $this->convertUserDateToAdsDate($minimumDate);

        if (empty($minimumDate)) {
            return;
        }

        $this->minimumDate = $minimumDate;
    }
}